# 卡密系统 API 文档

> 版本：v1.0
> 更新日期：2026-02-03

---

## 概述

卡密系统提供RESTful API接口，支持：
- 用户注册与登录
- 卡密激活与验证
- 设备绑定与管理

所有接口均需要API Key认证。

---

## 基础信息

| 项目 | 值 |
|-----|-----|
| API地址 | https://your-domain.com/api.php |
| 请求方式 | POST (JSON) |
| 认证方式 | Bearer Token |
| 字符编码 | UTF-8 |

---

## 认证方式

在请求头中添加Authorization字段：

```
Authorization: Bearer YOUR_API_KEY
```

---

## 接口列表

### 1. 用户注册

**接口地址**：`?action=register`

**请求参数**：

| 参数 | 类型 | 必填 | 说明 |
|-----|------|-----|------|
| phone | string | 是 | 手机号 |
| password | string | 是 | 密码（至少6位） |
| device_id | string | 是 | 设备ID |

**请求示例**：
```json
{
    "action": "register",
    "phone": "13800138000",
    "password": "your_password",
    "device_id": "dev_xxxxxxxxxxxxxx"
}
```

**返回示例**：
```json
{
    "success": true,
    "message": "注册成功",
    "data": {
        "user_id": 1001,
        "phone": "13800138000"
    }
}
```

---

### 2. 用户登录

**接口地址**：`?action=login`

**请求参数**：

| 参数 | 类型 | 必填 | 说明 |
|-----|------|-----|------|
| phone | string | 是 | 手机号 |
| password | string | 是 | 密码 |
| device_id | string | 是 | 设备ID |

**请求示例**：
```json
{
    "action": "login",
    "phone": "13800138000",
    "password": "your_password",
    "device_id": "dev_xxxxxxxxxxxxxx"
}
```

**返回示例**：
```json
{
    "success": true,
    "message": "登录成功",
    "data": {
        "token": "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9...",
        "expire_at": 1735689600
    }
}
```

---

### 3. 获取用户信息

**接口地址**：`?action=get_user_info`

**请求参数**：

| 参数 | 类型 | 必填 | 说明 |
|-----|------|-----|------|
| user_id | int | 是 | 用户ID |

**请求示例**：
```json
{
    "action": "get_user_info",
    "user_id": 1001
}
```

**返回示例**：
```json
{
    "success": true,
    "data": {
        "user_id": 1001,
        "phone": "13800138000",
        "vip_level": 1,
        "expire_date": "2026-02-28 23:59:59",
        "created_at": "2026-01-01 00:00:00"
    }
}
```

---

### 4. 激活卡密

**接口地址**：`?action=activate`

**请求参数**：

| 参数 | 类型 | 必填 | 说明 |
|-----|------|-----|------|
| card_key | string | 是 | 卡密 |
| device_id | string | 是 | 设备ID |

**请求示例**：
```json
{
    "action": "activate",
    "card_key": "KMXXXXXXXXXXXXXX",
    "device_id": "dev_xxxxxxxxxxxxxx"
}
```

**返回示例**：
```json
{
    "success": true,
    "message": "卡密激活成功",
    "data": {
        "expire_date": "2026-02-28 23:59:59",
        "max_devices": 3,
        "devices_bound": 1
    }
}
```

---

### 5. 验证卡密

**接口地址**：`?action=verify`

**请求参数**：

| 参数 | 类型 | 必填 | 说明 |
|-----|------|-----|------|
| card_key | string | 是 | 卡密 |

**请求示例**：
```json
{
    "action": "verify",
    "card_key": "KMXXXXXXXXXXXXXX"
}
```

**返回示例**：
```json
{
    "success": true,
    "data": {
        "valid": true,
        "status": "activated",
        "expire_date": "2026-02-28 23:59:59",
        "remaining_days": 25
    }
}
```

---

### 6. 心跳检测

**接口地址**：`?action=heartbeat`

**请求参数**：

| 参数 | 类型 | 必填 | 说明 |
|-----|------|-----|------|
| card_key | string | 是 | 卡密 |
| device_id | string | 是 | 设备ID |

**请求示例**：
```json
{
    "action": "heartbeat",
    "card_key": "KMXXXXXXXXXXXXXX",
    "device_id": "dev_xxxxxxxxxxxxxx"
}
```

**返回示例**：
```json
{
    "success": true,
    "message": "心跳正常",
    "data": {
        "expire_date": "2026-02-28 23:59:59"
    }
}
```

---

### 7. 获取卡密信息

**接口地址**：`?action=get_card_info`

**请求参数**：

| 参数 | 类型 | 必填 | 说明 |
|-----|------|-----|------|
| card_key | string | 是 | 卡密 |

**请求示例**：
```json
{
    "action": "get_card_info",
    "card_key": "KMXXXXXXXXXXXXXX"
}
```

**返回示例**：
```json
{
    "success": true,
    "data": {
        "card_key": "KMXXXXXXXXXXXXXX",
        "status": "activated",
        "card_type": "time",
        "duration": 30,
        "expire_date": "2026-02-28 23:59:59",
        "max_use_count": 1,
        "bound_devices": 1,
        "activated_at": "2026-01-01 00:00:00"
    }
}
```

---

## 错误码说明

| 错误码 | 说明 |
|-------|------|
| 1001 | 卡密不存在 |
| 1002 | 卡密已禁用 |
| 1003 | 卡密已过期 |
| 1004 | 设备数已达上限 |
| 1005 | 设备未绑定 |
| 2001 | 用户不存在 |
| 2002 | 密码错误 |
| 2003 | 用户已被禁用 |
| 3001 | API Key无效 |
| 3002 | 请求参数错误 |

**错误返回格式**：
```json
{
    "success": false,
    "message": "卡密不存在或已禁用",
    "error_code": 1001
}
```

---

## 完整对接示例

### Python示例 (python-example.py)

```python
# -*- coding: utf-8 -*-
import requests
import json

API_URL = "https://your-domain.com/api.php"
API_KEY = "your-api-key-here"

def make_request(action, data):
    """发送API请求"""
    headers = {
        "Content-Type": "application/json",
        "Authorization": f"Bearer {API_KEY}"
    }
    full_data = {"action": action, **data}
    response = requests.post(API_URL, json=full_data, headers=headers)
    return response.json()

def register_user(phone, password, device_id):
    """用户注册"""
    return make_request("register", {
        "phone": phone,
        "password": password,
        "device_id": device_id
    })

def login(phone, password, device_id):
    """用户登录"""
    return make_request("login", {
        "phone": phone,
        "password": password,
        "device_id": device_id
    })

def activate_card(card_key, device_id):
    """激活卡密"""
    return make_request("activate", {
        "card_key": card_key,
        "device_id": device_id
    })

def verify_card(card_key):
    """验证卡密"""
    return make_request("verify", {
        "card_key": card_key
    })

def heartbeat(card_key, device_id):
    """心跳检测"""
    return make_request("heartbeat", {
        "card_key": card_key,
        "device_id": device_id
    })

# 使用示例
if __name__ == "__main__":
    device_id = f"dev_{int(time.time())}"

    # 注册用户
    result = register_user("13800138000", "password123", device_id)
    print(f"注册: {result}")

    # 激活卡密
    result = activate_card("KMXXXXXXXXXXXXXX", device_id)
    print(f"激活: {result}")
```

### JavaScript示例 (javascript-example.js)

```javascript
// API配置
const API_URL = 'https://your-domain.com/api.php';
const API_KEY = 'your-api-key-here';

/**
 * 发送API请求
 */
async function apiRequest(action, data) {
    const response = await fetch(API_URL, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${API_KEY}`
        },
        body: JSON.stringify({ action, ...data })
    });
    return response.json();
}

/**
 * 用户注册
 */
async function register(phone, password, deviceId) {
    return apiRequest('register', { phone, password, device_id: deviceId });
}

/**
 * 用户登录
 */
async function login(phone, password, deviceId) {
    return apiRequest('login', { phone, password, device_id: deviceId });
}

/**
 * 激活卡密
 */
async function activateCard(cardKey, deviceId) {
    return apiRequest('activate', { card_key: cardKey, device_id: deviceId });
}

/**
 * 验证卡密
 */
async function verifyCard(cardKey) {
    return apiRequest('verify', { card_key: cardKey });
}

/**
 * 心跳检测
 */
async function heartbeat(cardKey, deviceId) {
    return apiRequest('heartbeat', { card_key: cardKey, device_id: deviceId });
}

// 使用示例
const deviceId = 'dev_' + Date.now();

activateCard('KMXXXXXXXXXXXXXX', deviceId)
    .then(result => console.log('激活结果:', result))
    .catch(err => console.error('激活失败:', err));
```

### cURL命令 (curl-commands.txt)

```bash
# 设置变量
API_URL="https://your-domain.com/api.php"
API_KEY="your-api-key-here"

# 激活卡密
curl -X POST "$API_URL" \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer $API_KEY" \
  -d '{
    "action": "activate",
    "card_key": "KMXXXXXXXXXXXXXX",
    "device_id": "dev_xxxxxxxxxxxxxx"
  }'

# 验证卡密
curl -X POST "$API_URL" \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer $API_KEY" \
  -d '{
    "action": "verify",
    "card_key": "KMXXXXXXXXXXXXXX"
  }'

# 心跳检测
curl -X POST "$API_URL" \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer $API_KEY" \
  -d '{
    "action": "heartbeat",
    "card_key": "KMXXXXXXXXXXXXXX",
    "device_id": "dev_xxxxxxxxxxxxxx"
  }'
```

---

## 常见问题 (FAQ)

### Q1: 设备ID如何生成？

设备ID由前端生成，建议使用以下格式：
```
dev_ + 8位随机字符串
```

示例：`dev_a1b2c3d4`

### Q2: API调用频率限制？

当前版本暂无限制，但建议：
- 心跳检测：每5分钟1次
- 其他接口：按需调用

### Q3: Token过期时间？

Token有效期为30天，过期后需要重新登录获取。

### Q4: 如何处理并发请求？

系统支持并发请求，但同一卡密的激活操作会加锁处理。

### Q5: 设备绑定数量？

默认最多绑定3台设备，可在后台配置。

---

## 技术支持

如有问题，请联系：
- 邮箱：support@example.com
- 小红书：@乐意梨

---

*文档最后更新时间：2026-02-03*
