/**
 * 卡密系统 JavaScript 对接示例
 *
 * 使用方法：
 * 1. 修改 CONFIG 中的 API_URL 和 API_KEY
 * 2. 在你的项目中引入此文件
 * 3. 调用 CardSystemAPI 类的方法
 *
 * 作者: 乐意梨
 * 日期: 2026-02-03
 */

// ============ 配置区域 ============
const CONFIG = {
    API_URL: 'https://your-domain.com/api.php',  // 替换为你的API地址
    API_KEY: 'your-api-key-here'                  // 替换为你的API Key
};
// ==================================

/**
 * 卡密系统API封装类
 */
class CardSystemAPI {
    constructor() {
        this.apiUrl = CONFIG.API_URL;
        this.apiKey = CONFIG.API_KEY;
    }

    /**
     * 发送API请求
     * @param {string} action - 接口名称
     * @param {object} data - 请求数据
     * @returns {Promise<object>}
     */
    async request(action, data = {}) {
        const payload = { action, ...data };

        try {
            const response = await fetch(this.apiUrl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${this.apiKey}`
                },
                body: JSON.stringify(payload)
            });

            return await response.json();
        } catch (error) {
            return {
                success: false,
                message: `网络请求失败: ${error.message}`,
                error_code: 9999
            };
        }
    }

    /**
     * 用户注册
     * @param {string} phone - 手机号
     * @param {string} password - 密码
     * @param {string} deviceId - 设备ID
     */
    async register(phone, password, deviceId) {
        return this.request('register', { phone, password, device_id: deviceId });
    }

    /**
     * 用户登录
     * @param {string} phone - 手机号
     * @param {string} password - 密码
     * @param {string} deviceId - 设备ID
     */
    async login(phone, password, deviceId) {
        return this.request('login', { phone, password, device_id: deviceId });
    }

    /**
     * 获取用户信息
     * @param {number} userId - 用户ID
     */
    async getUserInfo(userId) {
        return this.request('get_user_info', { user_id: userId });
    }

    /**
     * 激活卡密
     * @param {string} cardKey - 卡密
     * @param {string} deviceId - 设备ID
     */
    async activate(cardKey, deviceId) {
        return this.request('activate', { card_key: cardKey, device_id: deviceId });
    }

    /**
     * 验证卡密
     * @param {string} cardKey - 卡密
     */
    async verify(cardKey) {
        return this.request('verify', { card_key: cardKey });
    }

    /**
     * 心跳检测
     * @param {string} cardKey - 卡密
     * @param {string} deviceId - 设备ID
     */
    async heartbeat(cardKey, deviceId) {
        return this.request('heartbeat', { card_key: cardKey, device_id: deviceId });
    }

    /**
     * 获取卡密信息
     * @param {string} cardKey - 卡密
     */
    async getCardInfo(cardKey) {
        return this.request('get_card_info', { card_key: cardKey });
    }
}

/**
 * 设备ID管理工具
 */
const DeviceManager = {
    /**
     * 获取或生成设备ID
     */
    getDeviceId() {
        let deviceId = localStorage.getItem('device_id');
        if (!deviceId) {
            deviceId = 'dev_' + Date.now().toString(16) + Math.random().toString(16).slice(2, 6);
            localStorage.setItem('device_id', deviceId);
        }
        return deviceId;
    },

    /**
     * 获取设备信息
     */
    getDeviceInfo() {
        const deviceId = this.getDeviceId();
        return {
            deviceId,
            userAgent: navigator.userAgent,
            language: navigator.language,
            platform: navigator.platform
        };
    }
};

/**
 * 前端对接示例
 */
class FrontendDemo {
    constructor() {
        this.api = new CardSystemAPI();
    }

    /**
     * 示例1: 激活卡密并跳转
     */
    async activateAndRedirect(inputCardKey) {
        const deviceId = DeviceManager.getDeviceId();
        const result = await this.api.activate(inputCardKey, deviceId);

        if (result.success) {
            alert('激活成功！即将跳转...');
            window.location.href = '/product';
        } else {
            alert(`激活失败：${result.message}`);
        }

        return result;
    }

    /**
     * 示例2: 验证订阅状态
     */
    async checkSubscription(inputCardKey) {
        const result = await this.api.verify(inputCardKey);

        if (result.success && result.data && result.data.valid) {
            // 订阅有效
            const remainingDays = result.data.remaining_days || 0;
            console.log(`订阅有效，剩余 ${remainingDays} 天`);
            this.showContent();
        } else {
            // 订阅无效
            alert('订阅已过期或无效，请续费');
            window.location.href = '/renew';
        }

        return result;
    }

    /**
     * 示例3: 启动心跳检测
     */
    startHeartbeat(inputCardKey) {
        const deviceId = DeviceManager.getDeviceId();

        // 立即执行一次
        this.api.heartbeat(inputCardKey, deviceId);

        // 每5分钟执行一次
        setInterval(async () => {
            const result = await this.api.heartbeat(inputCardKey, deviceId);
            if (!result.success) {
                console.error('心跳检测失败:', result.message);
            }
        }, 5 * 60 * 1000);
    }

    /**
     * 显示内容（订阅有效时调用）
     */
    showContent() {
        const content = document.getElementById('premium-content');
        if (content) {
            content.style.display = 'block';
        }

        const locked = document.getElementById('locked-content');
        if (locked) {
            locked.style.display = 'none';
        }
    }
}

// ============ 使用示例 ============

/**
 * 在页面加载时检查订阅
 */
document.addEventListener('DOMContentLoaded', async function() {
    // 从URL参数或本地存储获取卡密
    const urlParams = new URLSearchParams(window.location.search);
    const cardKey = urlParams.get('card') || localStorage.getItem('my_card_key');

    if (!cardKey) {
        document.getElementById('app').innerHTML = '<p>请先绑定卡密</p>';
        return;
    }

    // 保存卡密到本地
    localStorage.setItem('my_card_key', cardKey);

    // 检查订阅
    const demo = new FrontendDemo();
    await demo.checkSubscription(cardKey);

    // 启动心跳检测
    demo.startHeartbeat(cardKey);
});

/**
 * 激活表单提交处理
 */
async function handleActivate(event) {
    event.preventDefault();

    const cardKey = document.getElementById('card-key-input').value.trim();
    if (!cardKey) {
        alert('请输入卡密');
        return;
    }

    const demo = new FrontendDemo();
    await demo.activateAndRedirect(cardKey);
}

// 导出到全局
window.CardSystemAPI = CardSystemAPI;
window.DeviceManager = DeviceManager;
window.FrontendDemo = FrontendDemo;
