# -*- coding: utf-8 -*-
"""
卡密系统 Python 对接示例

使用方法：
1. 修改 API_URL 和 API_KEY 为你的实际配置
2. 运行脚本测试各接口

作者: 乐意梨
日期: 2026-02-03
"""

import requests
import json
import time

# ============ 配置区域 ============
API_URL = "https://your-domain.com/api.php"  # 替换为你的API地址
API_KEY = "your-api-key-here"                 # 替换为你的API Key
# ==================================


class CardSystemAPI:
    """卡密系统API封装类"""

    def __init__(self, api_url, api_key):
        self.api_url = api_url
        self.api_key = api_key
        self.headers = {
            "Content-Type": "application/json",
            "Authorization": f"Bearer {api_key}"
        }

    def _request(self, action, data=None):
        """发送API请求"""
        payload = {"action": action}
        if data:
            payload.update(data)

        try:
            response = requests.post(
                self.api_url,
                json=payload,
                headers=self.headers,
                timeout=30
            )
            return response.json()
        except requests.RequestException as e:
            return {
                "success": False,
                "message": f"网络请求失败: {str(e)}",
                "error_code": 9999
            }

    def register(self, phone, password, device_id):
        """用户注册"""
        return self._request("register", {
            "phone": phone,
            "password": password,
            "device_id": device_id
        })

    def login(self, phone, password, device_id):
        """用户登录"""
        return self._request("login", {
            "phone": phone,
            "password": password,
            "device_id": device_id
        })

    def get_user_info(self, user_id):
        """获取用户信息"""
        return self._request("get_user_info", {"user_id": user_id})

    def activate(self, card_key, device_id):
        """激活卡密"""
        return self._request("activate", {
            "card_key": card_key,
            "device_id": device_id
        })

    def verify(self, card_key):
        """验证卡密"""
        return self._request("verify", {"card_key": card_key})

    def heartbeat(self, card_key, device_id):
        """心跳检测"""
        return self._request("heartbeat", {
            "card_key": card_key,
            "device_id": device_id
        })

    def get_card_info(self, card_key):
        """获取卡密信息"""
        return self._request("get_card_info", {"card_key": card_key})


def generate_device_id():
    """生成设备ID"""
    return f"dev_{int(time.time()):08x}{''.join(str(i) for i in range(3))}"


def print_result(action, result):
    """打印结果"""
    print(f"\n{'='*50}")
    print(f"接口: {action}")
    print(f"结果: {json.dumps(result, ensure_ascii=False, indent=2)}")
    print(f"{'='*50}\n")


def main():
    """主函数 - 测试所有接口"""
    api = CardSystemAPI(API_URL, API_KEY)
    device_id = generate_device_id()

    print(f"测试设备ID: {device_id}")
    print(f"API地址: {API_URL}")

    # 测试卡密验证
    test_card_key = "KMXXXXXXXXXXXXXX"  # 替换为你的测试卡密

    print("\n" + "="*60)
    print("开始测试卡密系统API...")
    print("="*60)

    # 1. 验证卡密
    result = api.verify(test_card_key)
    print_result("verify (验证卡密)", result)

    # 2. 激活卡密
    result = api.activate(test_card_key, device_id)
    print_result("activate (激活卡密)", result)

    # 3. 心跳检测
    result = api.heartbeat(test_card_key, device_id)
    print_result("heartbeat (心跳检测)", result)

    # 4. 获取卡密信息
    result = api.get_card_info(test_card_key)
    print_result("get_card_info (获取卡密信息)", result)

    print("\n" + "="*60)
    print("测试完成！")
    print("="*60)


# 下面是前端对接的简化示例函数

def frontend_activate_card(card_key):
    """
    前端激活卡密示例

    在前端JavaScript中调用：
    async function activateCard() {
        const deviceId = getDeviceId();  // 获取设备ID
        const result = await api.activate(cardKey, deviceId);
        if (result.success) {
            // 激活成功，跳转到产品页面
            window.location.href = '/product';
        } else {
            // 激活失败，显示错误信息
            alert(result.message);
        }
    }
    """
    api = CardSystemAPI(API_URL, API_KEY)
    device_id = generate_device_id()
    return api.activate(card_key, device_id)


def frontend_verify_subscription(card_key):
    """
    前端验证订阅状态示例

    在页面加载时调用：
    async function checkSubscription() {
        const result = await api.verify(cardKey);
        if (result.success && result.data.valid) {
            // 订阅有效，显示内容
            showContent();
        } else {
            // 订阅无效，跳转到续费页面
            window.location.href = '/renew';
        }
    }
    """
    api = CardSystemAPI(API_URL, API_KEY)
    return api.verify(card_key)


if __name__ == "__main__":
    main()
